<?php
/**
 * Featured Image Dynamic Tag.
 *
 * @package BPFWE_Widgets
 * @since 1.0.0
 */

namespace BPFWE_Dynamic_Tag\Tags;

use Elementor\Controls_Manager;
use Elementor\Core\DynamicTags\Data_Tag;
use Elementor\Modules\DynamicTags\Module as TagsModule;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class Post_Featured_Image.
 *
 * Dynamic tag for retrieving the featured image of a post.
 *
 * @since 1.0.0
 */
class Post_Featured_Image extends Data_Tag {

	/**
	 * Get tag name.
	 *
	 * Retrieve the dynamic tag name for internal use.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Tag name.
	 */
	public function get_name() {
		return 'post-featured-image';
	}

	/**
	 * Get dynamic tag group.
	 *
	 * Retrieve the group the tag belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Dynamic tag group.
	 */
	public function get_group() {
		return 'post';
	}

	/**
	 * Get dynamic tag categories.
	 *
	 * Retrieve the list of categories the tag belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Dynamic tag categories.
	 */
	public function get_categories() {
		return [
			TagsModule::IMAGE_CATEGORY,
			TagsModule::MEDIA_CATEGORY,
			TagsModule::TEXT_CATEGORY,
		];
	}

	/**
	 * Get tag title.
	 *
	 * Retrieve the dynamic tag title displayed in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Tag title.
	 */
	public function get_title() {
		return esc_html__( 'Featured Image', 'better-post-filter-widgets-for-elementor' );
	}

	/**
	 * Get dynamic tag value.
	 *
	 * Retrieve the featured image data for the current post. If no featured image
	 * exists, the fallback value is returned.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $options Optional. Additional options.
	 * @return array Featured image data or fallback data.
	 */
	public function get_value( array $options = [] ) {
		$thumbnail_id = get_post_thumbnail_id();

		if ( $thumbnail_id ) {
			$image_url = wp_get_attachment_image_src( $thumbnail_id, 'full' );

			$image_data = [
				'id'  => $thumbnail_id,
				'url' => $image_url ? $image_url[0] : '',
			];
		} else {
			$image_data = $this->get_settings( 'fallback' );
		}

		return $image_data;
	}

	/**
	 * Register controls.
	 *
	 * Add controls for setting a fallback image if no featured image is available.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->add_control(
			'fallback',
			[
				'label' => esc_html__( 'Fallback', 'better-post-filter-widgets-for-elementor' ),
				'type'  => Controls_Manager::MEDIA,
			]
		);
	}
}
